//-----------------------------------------------------------------------------
// A Mandelbrot set drawing demo for C#Builder
//
// Form1.cs
//
// Example submitted by Valentino Kyriakides
//
//-----------------------------------------------------------------------------

using System;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Data;

namespace Mandelbrot
{
  /// <summary>
  /// Summary description for Form1.
  /// </summary>
  public class Form1 : System.Windows.Forms.Form
  {
    private ArrayList stack;             // Saves clippings and buffer
    private Rectangle RubberBand;        // The ruber band
    private bool stopPainting = false;   // A flag in order to stop painting
    Bitmap buffer;                       // A buffer for graphics output
    RectangleF section;                  // A clipping section
    
    public Form1()
    {
      InitializeComponent();
      stack = new ArrayList();
      buffer = new Bitmap(ClientRectangle.Size.Width, ClientRectangle.Size.Height);
    }

		#region Windows Form Designer generated code
    /// <summary>
    /// Required method for Designer support - do not modify
		/// the contents of this method with the code editor.
    /// </summary>
    private void InitializeComponent()
    {
		System.Resources.ResourceManager resources = new System.Resources.ResourceManager(typeof(Form1));
		// 
		// Form1
		// 
		this.AutoScaleBaseSize = new System.Drawing.Size(5, 13);
		this.ClientSize = new System.Drawing.Size(648, 534);
		this.Icon = ((System.Drawing.Icon)(resources.GetObject("$this.Icon")));
		this.Name = "Form1";
		this.Text = "Mandelbrot";
		this.Resize += new System.EventHandler(this.Form1_Resize);
		this.MouseDown += new System.Windows.Forms.MouseEventHandler(this.Form1_MouseDown);
		this.Load += new System.EventHandler(this.Form1_Load);
		this.MouseUp += new System.Windows.Forms.MouseEventHandler(this.Form1_MouseUp);
		this.MouseMove += new System.Windows.Forms.MouseEventHandler(this.Form1_MouseMove);

	}
		#endregion

    [STAThread]
    static void Main() 
    {
      Application.Run(new Form1());
    }

    protected override void OnPaint(System.Windows.Forms.PaintEventArgs e)
    {
      // Draws only the required clipping!
      e.Graphics.DrawImage(buffer,e.ClipRectangle, e.ClipRectangle, GraphicsUnit.Pixel); 
    }
    
    // Draws the Mandelbrot set
    private void Mandelbrot(Rectangle sz)
    {
      const double cLimit = 10000; 
      const double cMaxIterat = 256; 
      const float StartX = -2.1f;
      const float StartDX = 2.8f;
      if (section.IsEmpty)  // Initialization
      {
        section.X = StartX;
        section.Width = StartDX;
        section.Height = (float) sz.Height / sz.Width * StartDX;
        section.Y = -section.Height/2;
      }
      else                  // Calculate new clipping
      {
        section.X += section.Width * ((float) sz.X / ClientRectangle.Width);
        section.Y += section.Height * ((float) sz.Y / ClientRectangle.Height);
        section.Width *= (float)sz.Width/ClientRectangle.Width;
        section.Height*= (float)sz.Height/ClientRectangle.Height;
      }

      // Draw a white background
      Graphics g = Graphics.FromImage(buffer);
      g.FillRectangle(Brushes.White, ClientRectangle);
      stopPainting = false; 

      // Step widths
      double stepx = section.Width/(ClientRectangle.Width-1);
      double stepy = section.Height/(ClientRectangle.Height-1);
      int x = 0;    // Step counter
      int y = 0; 
      for(double zr = section.X; zr < section.X + section.Width; zr += stepx)
      {
        double r1 , re, im;
        for(double zi = section.Y; zi < section.Y + section.Height; zi += stepy)
        {
          re = 0;                          // Initialize real part
          im = 0;                          // Initialize imaginary part
          for(int it = 0; it < cMaxIterat; it ++)      // Point iteration
          {
            r1 = re * re - im * im + zr;
            im = 2 * re * im + zi;
            re = r1;
            if (Math.Abs(re) > cLimit || Math.Abs(im) > cLimit)
            {
              Color c = Color.FromArgb((int)(0xff000000 + (it * 2222)));
              buffer.SetPixel(x, y, c);    // Draw point 
              break;
            }
          }
          y++; 
        }
        y = 0; x++;
        Application.DoEvents();   // Allow event handling after each drawn line
        if (stopPainting)         // Abbort?
          return; 
        if((x+1) % 10 == 0)       // Draw new calculatet stripe
          Invalidate(new Rectangle((x-11), 0, 11,ClientRectangle.Height));
      }
      // Complete drawing
      Invalidate(new Rectangle((x-11),0, 11,ClientRectangle.Height));
    }

    private void Form1_Resize(object sender, System.EventArgs e)
    {
      if (ClientRectangle.IsEmpty)
        return; 
      Bitmap oldBuffer = buffer; 
      buffer = new Bitmap(ClientRectangle.Size.Width, ClientRectangle.Size.Height);
      Text = ClientRectangle.ToString();
      Mandelbrot(ClientRectangle);
    }

    private void Form1_Load(object sender, System.EventArgs e)
    {
      Show();
      Mandelbrot(ClientRectangle);
    }

    private void Form1_MouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
    {
      if ( (e.Button & MouseButtons.Left) > 0)  // Left mouse button down? 
      {
        RubberBand.Location = Control.MousePosition; 
      }
      // Right mouse button down? Then recall previous clipping section. 
      if ( stack.Count > 1 && e.Button == MouseButtons.Right)  
      {
        stopPainting = true; 
        buffer = (Bitmap) stack[0];    // Pop 
        section = (RectangleF) stack[1];
        stack.RemoveRange(0,2);
        Invalidate();
      }
    }

    // Rubber band selection
    private void Form1_MouseMove(object sender, MouseEventArgs e)
    {
      if ( (e.Button & MouseButtons.Left) > 0)  // Left mouse button moved?
      {
        if(this.ClientRectangle.Contains(new Point(e.X, e.Y)))
        {
          // Delete the rubber band
          ControlPaint.DrawReversibleFrame(RubberBand, this.BackColor, 
            FrameStyle.Dashed);
          RubberBand.Size = 
            (Size) Control.MousePosition - (Size) RubberBand.Location;
          // Preserve aspect ratio
          float ratio = (float)ClientRectangle.Height/ClientRectangle.Width;
          RubberBand.Height = (int)(RubberBand.Width * ratio);
          // Redraw rubber band
          ControlPaint.DrawReversibleFrame(RubberBand, this.BackColor,
            FrameStyle.Dashed);
        }
      }
    }

    private void Form1_MouseUp(object sender, MouseEventArgs e)
    {
      if ( (e.Button & MouseButtons.Left) > 0)  // Left mouse button released?      
      {
        // Delete the rubber band
        ControlPaint.DrawReversibleFrame(RubberBand, this.BackColor,
          FrameStyle.Dashed);
        Rectangle NewSection = RubberBand; 
        RubberBand.Size = new Size(0, 0);
        NewSection.Location = PointToClient(NewSection.Location);
        if(NewSection.Width > ClientRectangle.Width / 20) // Rectangle not to small?
        {
          // Push current bitmap und clipping section onto the stack
          stack.Insert(0, section);  // Push  
          stack.Insert(0, buffer);
          buffer = (Bitmap)buffer.Clone();
          // Redraw the fractal
          Mandelbrot(NewSection);
        }
      }
    }
    
    protected override void OnPaintBackground(PaintEventArgs e)
    {
      // Intentionally left empty
    }
  }
}
